#include <afx.h>
#include <afxtempl.h>
#include <afxtempl.h>

#include "resource.H"
#include "parse.h"
#include "parsesrc.h"

inline BOOL IsNameCharExt(char ch, const char *szExt)
{
	if(szExt)
		return (isNameChar(ch) || (strchr(szExt, ch) != NULL));

	return isNameChar(ch);
}

void CSrcToken::ResetStart(const char *szSrc)
{
	ASSERT(szSrc >= m_sz);

	if(*szSrc)
	{
		m_nLen = szSrc - m_sz;
		m_sz = szSrc;
	}
	else
	{
		Empty();
	}
}


void CSrcToken::ResetEnd(const char *szSrc)
{
	ASSERT(szSrc >= m_sz);

	if(*szSrc)
	{
		m_nLen = (szSrc - m_sz);
	}
	else
	{
		Empty();
	}
}


CSrcToken::CSrcToken(const char *sz, int nLen)
{
	m_sz = sz; 
	m_nLen = nLen;
	m_bMatched = FALSE;
}

void ClearSrcTokens(CSrcTokenArray &aTokens)
{
	int n = aTokens.GetSize();
	int i;
	
	for(i = 0; i < n; i++)
		delete aTokens[i];
}


//@func Match any token from a list.
//@parm Source string to compare against keyword list
//@parm Keyword list
//@rdesc Return 0 if no match; otherwise, returns the length of
// the matched token.

int MatchToken(const char *sz, const char **aszTokens)
{
	int i;
	int nLen;

	for(i = 0; aszTokens[i]; i++)
	{
		nLen = strlen(aszTokens[i]);

		if(_strnicmp(sz, aszTokens[i], nLen) == 0)
			return nLen;
	}

	return 0;
};


CSrcItem::~CSrcItem(void)
{
	int n = m_aTokens.GetSize();
	int i;

	for(i = 0; i < n; i++)
		delete m_aTokens[i];
}


BOOL CSrcItem::GetToken(int i, CString &s)
{
	ASSERT(i >= 0); 
	
	if(m_aTokens.GetSize() > i)
	{
		s = *m_aTokens[i];
		return TRUE;
	}

	return FALSE;
}
		

BOOL CSrcParens::GetPhrase(CString &sPhraseRet)
{
	if(m_aTokens.GetSize() == 0)
		return CSrcItem::GetPhrase(sPhraseRet);

	sPhraseRet = m_chOpen;
	sPhraseRet += chSpace;
	sPhraseRet += *m_aTokens[0];
	sPhraseRet += chSpace;
	sPhraseRet += m_chClose;

	return TRUE;
}

int CSrcParens::Match(const char *szSrc, const char *&szEnd, BOOL &bMatch)
{
	const char *szFirst;
	const char *szLast;

	bMatch = FALSE;

	if(*szSrc != m_chOpen)
		return 0;

	szEnd = MatchParen(szSrc, m_chClose, TRUE);
	if(*szEnd != m_chClose)
		return warnSrcFieldsIncomplete;

	szFirst = EatWhite(szSrc+1);
	szLast = TrimWhite(szEnd, szSrc);

	if(szFirst <= szLast)
		Add( szFirst, szLast-szFirst+1 );
	else
		Add( szFirst, 0);

	szEnd++;

	bMatch = TRUE;

	return 0;
}


//@mfunc Match a keyword list item. This consists of one or more
// tokens matching the list of keywords stored with this object.
//
//@rdesc TRUE if successful, FALSE otherwise.

int CSrcKeyList::Match(const char *szSrc, const char *&szEnd, BOOL &bMatch)
{
	int nRet;
	int nMatch = 0;

	// key - a language keyword
	
	bMatch = TRUE;

	while(bMatch == TRUE)
	{
		nRet = CSrcKey::Match(szSrc, szEnd, bMatch);
		if(nRet)
			return nRet;

		if(bMatch)
		{
			nMatch++;
			szSrc = EatWhite(szEnd);
		}
	}

	bMatch = (nMatch != 0);

	return 0;
}


BOOL CSrcKeyList::GetPhrase(CString &sPhraseRet)
{
	int i, n;

	if((n = m_aTokens.GetSize()) == 0)
		return CSrcItem::GetPhrase(sPhraseRet);

	sPhraseRet = *m_aTokens[0];

	for(i = 1; i < n; i++)
	{
		sPhraseRet += chSpace;
		sPhraseRet += *m_aTokens[i];
	}

	return TRUE;
}

//@mfunc Match a single keyword.

int CSrcKey::Match(const char *szSrc, const char *&szEnd, BOOL &bMatch)
{
	ASSERT(m_aszKeys);

	int nLen;

	if(nLen = MatchToken(szSrc, m_aszKeys))
	{
		Add( szSrc, nLen );

		szEnd = EatWhite(szSrc + nLen);
	}

	bMatch = (nLen != 0);

	return 0;
}


BOOL CSrcKey::GetPhrase(CString &sPhraseRet)
{
	if(m_aTokens.GetSize() == 0)
		return CSrcItem::GetPhrase(sPhraseRet);

	sPhraseRet = *m_aTokens[0];

	return TRUE;
}


//@mfunc Match a single keyword followed by an identifer.
//
//@rdesc Zero if success or an error code.


int CSrcKeyName::Match(const char *szSrc, const char *&szEnd, BOOL &bMatch)
{
	int nRet;

	// key - a language keyword
	
    nRet = CSrcKey::Match(szSrc, szEnd, bMatch);
    if(nRet)
        return nRet;

    // no keyword, no match.

    if(!bMatch)
        return 0;

    // From this point forward, everything needs to be perfect.
    // 
    // Advance to the next token - needs to be an identifer.

	const char *szName;

    szName = szSrc = EatWhite(szEnd);

	if(!isNameStart(*szSrc))
		return warnSrcFieldsIncomplete;

	// Advance to the end of the name.

    for(szSrc++ ; IsNameCharExt(*szSrc, m_szNameExt); szSrc++);

	Add(szName, szSrc - szName);

    szEnd = EatWhite(szSrc);

	return 0;
}



BOOL CSrcKeyName::GetPhrase(CString &sPhraseRet)
{
	if(m_aTokens.GetSize() != 2)
		return CSrcItem::GetPhrase(sPhraseRet);

	sPhraseRet = *m_aTokens[0];
	sPhraseRet += chSpace;
	sPhraseRet += *m_aTokens[1];

	return TRUE;
}


//@mfunc Match a single keyword followed by a list of identifiers,
// separated by a single-character separator stored in m_chSep.
//
//@rdesc Zero on success or warnSrcFieldsIncomplete if the keyword
// was found without a valid identifier list.

int CSrcKeyNameList::Match(const char *szSrc, const char *&szEnd, BOOL &bMatch)
{
	int nRet;

    // First need to complete a key name.

    nRet = CSrcKeyName::Match(szSrc, szEnd, bMatch);
	if(nRet)
		return nRet;

    if(!bMatch)
        return 0;

    // At this point we have parsed the keyword and the first identifier
    // in the list. Now parse a list of identifiers (each preceded by
    // the separator character).

	const char *szName;

    while(m_chListSep == *szEnd)
    {
        szSrc = EatWhite(++szEnd);

        // got to be an identifier character here

        if(!isNameStart(*szSrc))
            return warnSrcFieldsIncomplete;

		szName = szSrc;
    
        // Advance to the end of the name.
    
        for(szSrc++ ; IsNameCharExt(*szSrc, m_szNameExt); szSrc++);

		Add(szName, szSrc - szName);
    
        szEnd = EatWhite(szSrc);
    }

	return 0;
}



BOOL CSrcKeyNameList::GetPhrase(CString &sPhraseRet)
{
	int n, i;

	if((n = m_aTokens.GetSize()) < 2)
		return CSrcItem::GetPhrase(sPhraseRet);

	sPhraseRet = *m_aTokens[0];
	sPhraseRet += chSpace;
	sPhraseRet += *m_aTokens[1];
	
	for(i = 2; i < n; i++)
	{
		sPhraseRet += m_chListSep;
		sPhraseRet += chSpace;
		sPhraseRet += *m_aTokens[1];
	}
	

	return TRUE;
}


BOOL CSrcName::GetPhrase(CString &sPhraseRet)
{
	if(m_aTokens.GetSize() == 0)
		return FALSE;

	sPhraseRet = *m_aTokens[0];

	return TRUE;
}


int CSrcName::Complete(CSrcToken &tokIn, CSrcToken &tokOut, BOOL &bMatch)
{
	const char *szSrc = tokIn.m_sz;
	int i = 0;

	bMatch = FALSE;
	tokOut = tokIn;

	if(!isNameStart(*szSrc))
	{
		if(m_bOptional)
		{
			return 0;
		}
		else
        {
			return warnSrcFieldsIncomplete;
        }
	}

	const char *szTok = szSrc;

	// Get the rest of the token.
	do {
			
		szSrc++, i++;
	
	} while(IsNameCharExt(*szSrc, m_szNameExt) && i < tokIn.m_nLen);

	Add(szTok, szSrc - szTok);
	
	szSrc = EatWhite(szSrc);

	bMatch = TRUE;

	tokOut.ResetStart(szSrc);

	return 0;
}


BOOL CSrcNameList::GetPhrase(CString &sPhraseRet)
{
	int i, n;
	
	if((n = m_aTokens.GetSize()) == 0)
		return FALSE;

	sPhraseRet = *m_aTokens[0];
	
	for(i = 1; i < n; i++)
	{
		sPhraseRet += chSpace;
		sPhraseRet += *m_aTokens[0];
	}

	return TRUE;
}

int CSrcNameList::Complete(CSrcToken &tokIn, CSrcToken &tokOut, BOOL &bMatch)
{
	int nRet;

	BOOL bMatchCur;

	bMatch = FALSE;

	// keep getting tokens until we reach the limit.

	do
	{
		nRet = CSrcName::Complete(tokIn, tokOut, bMatchCur);
		if(nRet)
			return nRet;

		if( bMatchCur )
			bMatch = TRUE;

		tokIn = tokOut;
	
	} while( bMatchCur && !tokIn.IsEmpty() );

	return 0;
}





int LinkTokens(
	const char *szSrc,
	const char *&szCont,
	CSrcItemList &lSrcItems, 
	CSrcTokenArray &aSrcTokens, 
	const char *szTermChars)
{
TRY
{
	// First verify that we have the entire declaration. If the
	// terminator character isn't here, append the next source line.
	
	const char *szTerm;
	if(szTermChars)
	{
		if( !(szTerm = strpbrk(szSrc, szTermChars)))
			return warnSrcFieldsMultiline;
	}
	else
	{
		szTerm = szSrc + strlen(szSrc);
	}

	// Now we should have the complete source declaration.
	// Now we walk through each "matchable" source item (each one that has
	// some specific phrase or character to search for, as opposed to generic
	// identifiers or identifer lists that don't have any fixed keywords
	// or any other way to distinguish them from other tokens).
	//
	// Every time we match a source item, disqualify that one (and all previous
	// ones) for later consideration (advance the list iterator).
	//
	// Each time through the loop we advance to the next source token. We
	// only store the token if it's not a continuation of an unidentified token
	// from the last loop.

	POSITION pos = lSrcItems.GetHeadPosition();
	POSITION posSrch;
	
	CSrcItem  *pItem;
	CSrcToken *pTok;

	const char *szSrch = EatWhite(szSrc);

	int iCurToken;
	int iLastUnmatchedToken = -100;
	int nRet;
	BOOL bMatch;

	iCurToken = 0;

	while( pos != NULL && szSrch < szTerm )
	{
		// Starting with the current source item, search forward through
		// the source item list to see if any of them match this source
		// token. We can only match tokens that start with a specific keyword or 
		// character. If it matches, create a source token and add it to the
		// list, and tell the source item to remember which token matched.
		
		bMatch  = FALSE;
		posSrch = pos;
		while(posSrch != NULL)
		{
			pItem = lSrcItems.GetNext( posSrch );

			// is this something we can match definitively - if it is
			// an identifier (not a specific keyword) we can't match 
			// it. If it matches, we can advance to the next source token
			// (end the search now).

			if(pItem->CanMatchDefinite())
			{
				nRet = pItem->Match( szSrch, szCont, bMatch );
				if(nRet)
					return nRet;

				if(bMatch)
				{
					pTok = new CSrcToken(szSrch, szCont-szSrch);
					
					pTok->SetMatch();
#ifdef _TRACEPARSE
					CString sToken(pTok->m_sz, pTok->m_nLen);
					TRACE("Matched token: %s\n", (const char *)sToken);
#endif
					aSrcTokens.Add( pTok );

					iCurToken++;

					// if we had a match, continue the remaining
					// search at the one following this one.
					
					pos = posSrch;

					pItem->SetMatch(pTok);

					break;
				}
			}
		}

		// if nothing matched, create a token and add it to the token
		// list. append the new token to the previous source token if the
		// last one didn't match either.

		if( ! bMatch )
		{
			// treat consecutive unmatched tokens as the same one
			// retrieve the last token and append the current token's
			// text to it.

			szCont = SkipIdentifier(szSrch);

			if(iLastUnmatchedToken == iCurToken-1)
			{
				pTok = aSrcTokens[iCurToken-1];

				pTok->m_nLen = szCont - pTok->m_sz;
#ifdef _TRACEPARSE
				CString sToken(pTok->m_sz, pTok->m_nLen);
				TRACE("Extended unmatched token: %s\n", (const char *)sToken);
#endif
			}
			else
			{
				pTok = new CSrcToken(szSrch, szCont-szSrch);
#ifdef _TRACEPARSE
				CString sToken(pTok->m_sz, pTok->m_nLen);
				TRACE("Unmatched token: %s\n", (const char *)sToken);
#endif
				aSrcTokens.SetAtGrow(iCurToken, pTok );

				iLastUnmatchedToken = iCurToken;

				iCurToken++;
			}

			// Advance to the next token.

			lSrcItems.GetNext( pos );
		}

		// szCont was updated either in the case of a match or in the
		// case of no match.

		szSrch = EatWhite(szCont);
	}
	
	szCont = szSrch;

    return 0;
}
CATCH(CMemoryException, e)
{
	return errMemory;
}
END_CATCH
}



//@func This function completes the parsing of a source line.
// The source line has already been tokenized and matched to all the
// keywords that are directly matchable via a keyword or character
// match. This function now goes through the list of targeted source
// items and checks to see:
//
// * If required and "matchable" items were already matched
//
// * If it can match more free form items (like identifiers or identifier
//   lists) from the leftover tokens that were not already matched
//
// * Finally, it needs to make sure there are no leftover, unused tokens
//   left in the source line.
//
//@rdesc Returns zero if everything parses alright or the expected
// error codes if something is wrong.
//

int CompleteTokens(
    CSrcItemList &lSrcItems,        //@parm List of source items already matched via <f LinkTokens>.
    CSrcTokenArray &aSrcTokens)     //@parm Token list produced by <f LinkTokens>.
{
    POSITION pos = lSrcItems.GetHeadPosition();
	POSITION posSearch;
	CSrcItem *pItem;
    int iSrcToken = 0;				// which source token we're looking at
	CSrcToken tokRet;
	CSrcToken *pNewTok;

	while( pos != NULL )
	{
		pItem = lSrcItems.GetNext( pos );

		// Item is pre-matchable - make sure it was already matched if it
		// is required. If it was matched, advance to the next source
		// token.

		if( pItem->CanMatchDefinite() )
		{
			CSrcToken *pTokenItem;			// which source token corresponds to a matched item
			if( pItem->GetMatch( pTokenItem ) )
			{
				// If the matched item isn't the same as the
				// current source token, we have some unidentified
				// text here, return error.

				if(*pTokenItem != *aSrcTokens[iSrcToken])
					return warnSrcFieldsUnexpected;

				// Advance to the next source token, done with this
				// loop iteration.

				iSrcToken++;
			}
			else
			{
				// Not matched, required, error.

				if( pItem->IsOptional() == FALSE )
					return warnSrcFieldsIncomplete;

				// Fall through to next iteration - we will now test
				// the next source item against the current source
				// token.
			}
		}
		
		// indefinite match, we are going to try to complete it now.

		else
		{
			posSearch = pos;
			
            // is the current source token already taken by a later item?
			
			if(aSrcTokens[iSrcToken]->IsMatched())
			{
				// if it is, and the source item is optional, we are done with
				// this iteration and can move to the next source token.
				// But if the source item is required, we have an error.

				if( pItem->IsOptional() )
				{
					iSrcToken++;
				}
				else
				{
					return warnSrcFieldsIncomplete;
				}
			}
			else
			{
				// It's not taken, so try to match it with the current item.
				// If it matches, advance to the next token.

				BOOL bMatch;
				int  nRet;

				nRet = pItem->Complete(*(aSrcTokens[iSrcToken]), tokRet, bMatch);
				if(nRet)
					return nRet;

				if(bMatch)
				{
					pItem->SetMatch(aSrcTokens[iSrcToken]);

					if( ! tokRet.IsEmpty() )
					{
						aSrcTokens[iSrcToken]->ResetEnd(tokRet.m_sz);

						pNewTok = new CSrcToken(tokRet.m_sz, tokRet.m_nLen);

						aSrcTokens.InsertAt(iSrcToken+1, pNewTok);
					}
					iSrcToken++;
				}
			}
		}
	}

    return 0;
}


